// Base64Codec.cpp: implementation of the Base64Codec class.
//
// (c) 2008 Tom A. Wheeler
//
//////////////////////////////////////////////////////////////////////


#include "stdafx.h"

#include "Base64Codec.h"

#include <string.h>



//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Base64Codec::Base64Codec()
{

}

Base64Codec::~Base64Codec()
{

}


//////////////////////////////////////////////////////////////////////
//
// Provide an estimate of the buffer size needed for a Base64 encode.
//
// Parameter: nInputLength, the number of octets in the input stream.
//
// Return value: The number of Base64 characters needed to represent
//				the stream, not counting the '\0' terminator (if used).
//
//////////////////////////////////////////////////////////////////////

int Base64Codec::EstimateBase64EncodeSize(int nInputLength)
{
int nLengthResidue;		// Residual number of bits that will remain after encoding. ZERO means
						// that input string contains an integral multiple of 6 data bits.

int nCharactersNeeded;	// Number of base64 characters needed to encode the data, including any
						// padding at the end.

// Assess how many characters are needed

nLengthResidue =   (nInputLength*8) % 6;

if (nLengthResidue == 0)
	nCharactersNeeded = (nInputLength*8)/6;
	else
	nCharactersNeeded = (nInputLength*8)/6 + 1;

// Messages must be a multiple of 4 characters. This code accounts for padding that might
// be appended to the end of the Base64 string.

if ((nCharactersNeeded % 4) != 0)
	nCharactersNeeded = 4*(nCharactersNeeded/4)+4;

return nCharactersNeeded;
}



//////////////////////////////////////////////////////////////////////
//
// Look-up table for encoding. From RFC3548, Page 4. The '==' sequence at the end is not part
// of the table and is present to add a protective margin against buffer overrun.
//
//////////////////////////////////////////////////////////////////////

char* Base64Codec::LUT = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/==";


//////////////////////////////////////////////////////////////////////
//
// Encode a set of binary data into a Base64 string. The data is strictly
// interpreted as binary; any values may be sent.
// 
// Parameters:	szOutBuf	--> A place where the Base64 encoded string will be written.
//				nOutbufsize --> Size of the buffer, NOT including the '\0' delimiter which will be added.
//				szInputString	--> The binary input data to be encoded.
//				nDataLength	--> The number of bytes in the binary input set.
//
// Returns: The length of the Base64 ASCII string, as encoded according to RFC3548, if successful, or
//			(-1) to signal a buffer overrun error (no data is written to the buffer in this case).
//
//////////////////////////////////////////////////////////////////////


int Base64Codec::Encode(char *szOutBuf, int nOutbufsize, const char* szInputString, int nDataLength)
{
int nLengthResidue;		// Residual number of bits that will remain after encoding. ZERO means
						// that input string contains an integral multiple of 6 data bits.

int nCharactersNeeded;	// Number of base64 characters needed to encode the data, including any
						// padding at the end.

int nLength;			// Length of encoded string
char cTemp;				// Used to store individual encoded characters.


// Assess how many characters are needed in the szOutbuf; if a buffer overrun will occur, then
// return an error code.

nLengthResidue =   (nDataLength*8) % 6;

if (nLengthResidue == 0)
	nCharactersNeeded = (nDataLength*8)/6;
	else
	nCharactersNeeded = (nDataLength*8)/6 + 1;

// Messages must be a multiple of 4 characters. This code accounts for padding that might
// be appended to the end of the Base64 string.

if ((nCharactersNeeded % 4) != 0)
	nCharactersNeeded = 4*(nCharactersNeeded/4)+4;


if (nCharactersNeeded > nOutbufsize)
	return (-1);	// Failure: A buffer overrun would occur


//
// Actual encoding process. The data is virtually buffered into a bit-stream.
// The LoadInternalBuffer() call merely initializes the pointers and counters into this
// stream.
//

LoadInternalBuffer(szInputString, nDataLength);	// Initialize pointers to data and bit counters

//
// Retrieve successive groups of six (6) bits from the octet stream. The GetSixBitsFromBuffer() method
// performs the actual work, including the insertion of '=' padding characters as needed at the end of
// the data stream.
//

for(nLength=0;;nLength++)
	{
	szOutBuf[nLength] = 0;

	if ((cTemp = GetSixBitsFromBuffer()) == 0)	// End of data stream?
			{
			break;
			}

	szOutBuf[nLength] = cTemp;				// Store Base64 encoded value
	}


return nLength;
}


//////////////////////////////////////////////////////////////////////
//
// LoadInternalBuffer()
//
// Parameters:	data		--> A pointer to the binary input stream
//				nByteCount	--> The number of bytes in the octet stream
//
// Action: No actual buffer is loaded; the internal variables for keeping track
// of the bit count, current datum pointer, and next bit position are initialized
// by this routine.
//
//////////////////////////////////////////////////////////////////////

void Base64Codec::LoadInternalBuffer(const char *data, int nByteCount)
{
m_nBufferBitCount = nByteCount * 8;
m_pCurrentBufferDatum = data;
m_nBufferNextBitPosition = 0;
}

//////////////////////////////////////////////////////////////////////
//
// GetSixBitsFromBuffer()
//
// No Parameters are passed, but LoadInternalBuffer() MUST be called first.
//
// Get the next six bits from the internal "buffer." This actually isn't a buffer at all; it's
// merely a pointer to the user's data input. The Base64 representation of the bits is returned,
// or binary zero (0) if no more bits are in the buffer. Padding characters are automatically
// inserted at the end of the data stream as needed.
//
//////////////////////////////////////////////////////////////////////

char Base64Codec::GetSixBitsFromBuffer()
{
unsigned char cResult = 0;
int nSubPosition;
int nNumBitsRemaining;

//
// The following logic inserts '=' padding characters at the end of the data, and
// also signals to the caller when no more data are available.
//

if (m_nBufferNextBitPosition >= m_nBufferBitCount)
		{
		if ((m_nBufferNextBitPosition % 8) == 0)
			{
			return 0;	// The real end of the data is here -- we reached a byte boundary.
			}
			else		// We're not quite to a byte boundary yet, so padding must be inserted.
			{
			m_nBufferNextBitPosition+=6;
			return '=';	// Return a padding character
			}
		}

//
// Calculate subposition, which is the bit-number within the current datum within the octet stream.
// The subposition may only have the legal values of 0, 6, 4, or 2, since these positions represent
// all possible residues modulo eight of the integer multiples of six.
//

nSubPosition = m_nBufferNextBitPosition % 8;	// Next place to read from in current buffer byte
nNumBitsRemaining = m_nBufferBitCount - m_nBufferNextBitPosition;	// Number of bits remaining

//
// Fetch the bit values at each subposition, spanning to the next octect (if possible).
// The resulting cResult value from the fetch will be an index into the LUT.
//

switch(nSubPosition)
	{
	case 0: cResult = ((*m_pCurrentBufferDatum) >> 2) & 0x3f;
			break;


	case 6:	if (nNumBitsRemaining > 8)
				cResult = (((*m_pCurrentBufferDatum) << 4) & 0x30) | ((m_pCurrentBufferDatum[1] >> 4) & 0xf );
				else
				cResult = ((*m_pCurrentBufferDatum) << 4) & 0x30;	// Pad low bits with zero on last character

			m_pCurrentBufferDatum++;	// will need to span to next character in sequence
			break;

	case 4:	if (nNumBitsRemaining > 8)
				cResult = ( ((*m_pCurrentBufferDatum) << 2) & 0x3c) | ((m_pCurrentBufferDatum[1] >> 6) & 0x3 );
				else
				cResult = ((*m_pCurrentBufferDatum) << 2) & 0x3c;

			m_pCurrentBufferDatum++;	// will need to span to next character in sequence
			break;

	case 2: if (nNumBitsRemaining >= 6)
				cResult = (*m_pCurrentBufferDatum) & 0x3f;
				else
				return 0;

			m_pCurrentBufferDatum++;	// will need to span to next character in sequence
			break;


	default: return 0;	// Some kind of error has occured. The only possible subpositions are 0, 6, 4, and 2.
	}


m_nBufferNextBitPosition+=6;	// Address next group of six bits for next call to this routine

return( LUT[cResult % 64] );	// Return the Base64 representation for this group of six bits.
}



//////////////////////////////////////////////////////////////////////
//
// Decode a sextet of data.
// 
// Parameter:	cDatum	--> The Base64 character representing the six bits of data.
//				
// Returns: The lower six bits of the integer value contain the binary value of the datum.
//			If the passed symbol is invalid, (-1) is returned to indicate the error.
//
//////////////////////////////////////////////////////////////////////

int Base64Codec::DecodeSextet(char cDatum)
{
int i;
char* pTable = LUT;


if (cDatum == '=') return 0;	// Padding

for(i=0 ; i<64 ; i++, pTable++)
	{
	if ( *pTable == cDatum )
		return i;
	}

return (-1);	// Invalid symbol (not in table)
}




//////////////////////////////////////////////////////////////////////
//
// Decode a Base64 string into a set of binary data.
// 
// Parameters:	OutBuf	--> A place where the Base64 encoded string will be written.
//				nOutbufsize --> Size of the buffer, NOT including the '\0' delimiter which will be added.
//				szInputString	--> The Base64 input data to be decoded.
//				
// Returns: The number of bytes written to the output buffer, or (-1) if a decode
//			error occurs.
//
//////////////////////////////////////////////////////////////////////

int Base64Codec::Decode(char* OutBuf, int nOutbufsize, const char *szInputString)
{
int nMessageLength;
int nPadCharCount;
int nOctetCount;
int i;
int nResult;
char* pCurrentDatum;

nMessageLength = strlen(szInputString);

if (( (nMessageLength % 4) != 0) || ( nMessageLength < 4))
		return(-1);			// Malformed Base64 message string

//
// Examine the message and determine the number of padding bits at the end of the message.
//

nPadCharCount = 0;

if ( szInputString[ nMessageLength - 1] == '=')
		nPadCharCount++;

if ( szInputString[ nMessageLength - 2] == '=')
		nPadCharCount++;

//
// Determine the actual number of octets in the byte stream
//


nOctetCount = (6*nMessageLength - 6*nPadCharCount) / 8;




//
// Loop through the input data string, peeling off the octets
//




for(i=0 , m_nBufferNextBitPosition = 0, pCurrentDatum = OutBuf ; i < nMessageLength ; i++)
	{
	nResult = DecodeSextet( szInputString[i] );

	if (nResult < 0) return (-1);	// Invalid symbol error


	// Decode sextets into the output buffer. They may be aligned at positions
	// 0, 6, 4, or 2.

	switch(m_nBufferNextBitPosition % 8)
			{

			case 0:
				*pCurrentDatum = 0xfc & (nResult << 2);
				break;

			case 6:
				*pCurrentDatum = (*pCurrentDatum) | (0x3 & (nResult >> 4) );
				pCurrentDatum++;
				*pCurrentDatum = 0xf0 & (nResult << 4);
				break;


			case 4:
				*pCurrentDatum = (*pCurrentDatum) | (0xf & (nResult >> 2) );
				pCurrentDatum++;
				*pCurrentDatum = 0xc0 & (nResult << 6);
				break;

			case 2:
				*pCurrentDatum = (*pCurrentDatum) | (0x3f & nResult);
				pCurrentDatum++;
				break;

			}

	m_nBufferNextBitPosition += 6;
	}


return nOctetCount;

}

//////////////////////////////////////////////////////////////////////
//
// Estimate the number of octets encoded within a Base64 data stream.
//
// Parameter: szInputString - A zero-terminated string of Base64 symbols.
//
// Returns: The number of octets, or (-1) if the data stream is invalid.
//
// WARNING: This function does not parse the data stream for improper
// symbols. Streams with invalid data may return a valid octet count.
//
//////////////////////////////////////////////////////////////////////

int Base64Codec::EstimateBase64DecodeSize(char *szInputString)
{
int nMessageLength;
int nPadCharCount;
int nOctetCount;

nMessageLength = strlen(szInputString);

if (( (nMessageLength % 4) != 0) || ( nMessageLength < 4))
		return(-1);			// Malformed Base64 message string

//
// Examine the message and determine the number of padding bits at the end of the message.
//

nPadCharCount = 0;

if ( szInputString[ nMessageLength - 1] == '=')
		nPadCharCount++;

if ( szInputString[ nMessageLength - 2] == '=')
		nPadCharCount++;

//
// Determine the actual number of octets in the byte stream
//


nOctetCount = (6*nMessageLength - 6*nPadCharCount) / 8;

return nOctetCount;
}
