// DatagramSocket.cpp: implementation of the DatagramSocket class.
//
//////////////////////////////////////////////////////////////////////

//#include <winsock2.h>	// MUST be first in list
#include "stdafx.h"
#include "DatagramSocket.h"



//////////////////////////////////////////////////////////////////////
// Symbolic Constants. These are returned by the various I/O
// routines.
//////////////////////////////////////////////////////////////////////

int DatagramSocket::SOCK_ERROR = SOCKET_ERROR;
int DatagramSocket::SOCK_TIMEOUT = SOCKET_ERROR - 1;
int DatagramSocket::SOCK_NODATA = 0;



//
// Global data element to record WSAInit state values
//

WSADATA DatagramSocket::m_Data;

//////////////////////////////////////////////////////////////////////
// Construction
//
// Passing no argument for "nPort" results in this being a transmitting
// UDP socket; passing in a non-zero port number results in the socket
// being bound to a port for receiving incoming datagrams.
//
// After construction, use GetStatus() to learn the object's status. Zero
// means the DatagramSocket did not successfully construct; non-zero means
// construction was successful.
//
//////////////////////////////////////////////////////////////////////

DatagramSocket::DatagramSocket(int nPort)
{
m_nStatus = 0;
m_PORT = nPort;

//
// If the port number is non-zero, attempt to bind to it and return the status.

if (nPort != 0)
	{
	m_nStatus = StartUDPReceiver(); // Try to listen; returns 0 on failure.

	if (m_nStatus == 0) return;		// BIND failure



	//
	// Success constructing this DatagramSocket!
	//

	m_nStatus = 1;
	return;
	}

//
// Port = 0 means that this is a transmitting DatagramSocket.
//

m_rcvSocket = socket(AF_INET, SOCK_DGRAM, 0);	// create a UDP socket (SOCK_DGRAM)

if (m_rcvSocket == INVALID_SOCKET)
	return;									// error creating socket

//
// Enable broadcasting on this socket
//

int nEnable = 1;	// 1=Enable broadcasting
::setsockopt(m_rcvSocket,SOL_SOCKET,SO_BROADCAST,(char*) &nEnable, sizeof(nEnable) );

m_nStatus = 1;
}


DatagramSocket::~DatagramSocket()
{
UnBind();
}

int DatagramSocket::WSAStartup()
{
return ::WSAStartup(MAKEWORD(1,1), &DatagramSocket::m_Data);
}

int DatagramSocket::GetStatus()
{
return m_nStatus;
}

int DatagramSocket::StartUDPReceiver()
{
int nStatus;

memset(&m_destSockAddr,0,sizeof(m_destSockAddr));
m_destSockAddr.sin_port = htons(m_PORT);
m_destSockAddr.sin_family = AF_INET;
m_destSockAddr.sin_addr.s_addr = htonl(INADDR_ANY);	// Any IP can talk to this socket

m_rcvSocket = socket(AF_INET, SOCK_DGRAM, 0);	// create a UDP socket (SOCK_DGRAM)

if (m_rcvSocket == INVALID_SOCKET)
	return 0;									// error creating socket

//
// Bind the UDP socket to the listening port
//

nStatus = bind(m_rcvSocket, (LPSOCKADDR) &m_destSockAddr,sizeof(m_destSockAddr));

if (nStatus == SOCKET_ERROR)
	return 0;									// error binding socket

return 1;	// OK, successfully bound this socket to a UDP port
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// Receive data from the UDP socket. Returns the number of bytes read, zero (no data), or
// an error code.
//
/////////////////////////////////////////////////////////////////////////////////////////

int DatagramSocket::receive(void *lpBuf, int nBufSize)
{
int nResult;
if (m_nStatus == 0) return SOCK_ERROR;
int nAddrSize;

nAddrSize = sizeof(m_sourceSockAddr);

nResult = ::recvfrom(m_rcvSocket,	// Receiving socket
						(char*) lpBuf,		// Buffer
						nBufSize,	// Buffer size
						0,			// No Flags Set
						(SOCKADDR*) &m_sourceSockAddr,
						&nAddrSize );

if (nResult != SOCKET_ERROR) return nResult;

// Got an error, is it an outright socket error or just a timeout?

nResult = WSAGetLastError();

if (nResult == WSAETIMEDOUT ) return SOCK_TIMEOUT;

return SOCK_ERROR;


}


void DatagramSocket::SetSoTimeout(int nMilli)
{
if (m_nStatus == 0) return; // Not open

::setsockopt(m_rcvSocket,SOL_SOCKET,SO_RCVTIMEO,
		(char*) &nMilli, sizeof(int));

::setsockopt(m_rcvSocket,SOL_SOCKET,SO_SNDTIMEO,
		(char*) &nMilli, sizeof(int));
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// Send data on the UDP socket. Returns the number of bytes sent, zero (no data), or
// an error code. This version of the method will perform a full DNS lookup if the
// destination host is specified using a FQDN; for better performance, pass the IP
// address directly.
//
// Returns: Number of bytes transmitted, or an error code. The number of bytes
// sent to the routine should not exceed the MTU of the data link layer device being
// utilized.
//
/////////////////////////////////////////////////////////////////////////////////////////


int DatagramSocket::send(void *lpBuf, int nBytes, char *szHost, int nDestPort)
{
hostent* remhost;
unsigned int addr = 0;		// Used to hold IP address (binary)
							// Will be 0xffffffff if a broadcast
							// Will be ZERO if DNS services were used; REMHOST will
							// hold the summarized DNS data in this case

int nResult;

if (m_nStatus == 0) return SOCK_ERROR;	// Error: This DatagramSocket never constructed
										// properly; it's unuseable.

//
// If the hostname starts with alpha character, use DNS services to resolve it.
//

if (isalpha(*szHost))
	{
	addr = 0;							// FLAG: We used DNS services
	remhost = ::gethostbyname(szHost);	// Use DNS to lookup the host.
	if (remhost == NULL) return SOCK_ERROR;		// ERROR: unable to do anything further	
	if (WSAGetLastError() == WSAHOST_NOT_FOUND ) return SOCK_ERROR; // ERROR: Host not found
	}
	else
//
// Otherwise, just convert the ASCII representation of the host IP into binary.
//
	{
	addr = inet_addr(szHost);		// Get the raw binary IP address
//	if ((remhost == NULL) && (addr != INADDR_BROADCAST)) return SOCK_ERROR;		// ERROR: unable to do anything further	
	}



//
// Convert the address data in the structure pointed by *remhost into a 
// SOCKADDR_IN structure for opening the socket.
// The SOCKADDR_IN structure holds the destination address, port, and address
// family identifier.
//
//

if (addr == 0) // We used DNS, so REMHOST has the IP address information 
	{
	if (remhost->h_addrtype != AF_INET ) return SOCK_ERROR;	// ERROR: Not an IP address, can't process

	if (remhost->h_addr_list[0] == NULL) return SOCK_ERROR; // ERROR: No address data returned
	//
	// Crude hack: We only use the first IP address returned by DNS. This might
	// cause problems for some users.
	//
	::memcpy(&m_destSockAddr.sin_addr , remhost->h_addr_list[0], remhost->h_length);
	}
	else
	// We didn't use DNS, we just got a raw IP address in binary
	::memcpy(&m_destSockAddr.sin_addr,(char*) &addr, sizeof(addr) );

//
// Set the destination port and address family information
//


m_destSockAddr.sin_port = htons(nDestPort);			// Destination port
m_destSockAddr.sin_family = AF_INET;		// Family = Internet (IP)

nResult = ::sendto(m_rcvSocket,				// Socket sending on
					(char*) lpBuf,
					nBytes,
					0,						// No flags set
					(SOCKADDR*) &m_destSockAddr,
					sizeof(m_destSockAddr) );


if (nResult == SOCKET_ERROR) return SOCK_ERROR;

return nResult;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// GetMTU()
//
// Reports the maximum number of bytes that can be transmitted by one call to the
// SENDTO() method of DatagramSocket.
//
// Be careful; this does not correspond to the MTU of the data link layer protcol.
// The operating system will typically break a large message into a number of datagram
// IP fragments if it's too large to fit into one transmit packet. This may or may not
// be correctly reassembled at the receiver, since IP does not guarantee delivery of
// packets.
//
// For Ethernet, data should be no more than about 1450 bytes per packet to prevent
// fragmentation of the message.
//
//
////////////////////////////////////////////////////////////////////////////////////////

int DatagramSocket::GetMTU()
{
int nMaxSize;
int nOptLen=sizeof(nMaxSize);

::getsockopt(m_rcvSocket,SOL_SOCKET,SO_MAX_MSG_SIZE,(char*) &nMaxSize, &nOptLen );

return nMaxSize;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// GetSenderAddress()
//
// Returns a new InetAddress object holding the address of the last message packet
// received on this datagram socket.
//
// Returns: <NULL> if no data received, or socket not successfully bound to a port.
//
//
////////////////////////////////////////////////////////////////////////////////////////

InetAddress* DatagramSocket::GetSenderAddress()
{
unsigned long dwBinaryIP;

if (m_nStatus == 0) return NULL;	// Error: Socket not initialized

if (m_PORT == 0) return NULL;		// Error: Not bound to a port number


::memcpy(&dwBinaryIP, &m_sourceSockAddr.sin_addr, sizeof(dwBinaryIP) );

return new InetAddress(dwBinaryIP );
}

///////////////////////////////////////////////////////////////////////////////////////
//
// UnBind()
//
// Cancels any binding between this DatagramSocket and a particular port number.
// Automatically called by the destructor, but can be called manually to disconnect
// the binding without destroying the DatagramSocket object.
//
// Returns: None
//
///////////////////////////////////////////////////////////////////////////////////////

void DatagramSocket::UnBind()
{
if (m_nStatus == 0)
	return;


shutdown(m_rcvSocket, 2);
closesocket( m_rcvSocket );		// Close the open, listening socket to free it up.

m_nStatus = 0;
}
