// Host.cpp: implementation of the CHost class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "Host.h"

//////////////////////////////////////////////////////////////////////
// Construction
//
// What is passed:
//
// pCallback: A pointer to the "TalkToClient" function provided by the user. (Required, may be NULL, which
//            will not cause an error, but it will also make the service not start!).
//
// nMaxClients: The maximum number of TCP clients to be supported.
// nTCPPort: The TCP Port number to listen on.
//
// Make sure WSAStartup() has been called before starting the service.
//
//////////////////////////////////////////////////////////////////////

CHost::CHost(void (*pCallback)(Socket& s, int& nHaltFlag), void (*pAcceptOnIdle)(), int nMaxClients, int nTCPPort)
{

m_pCallback = pCallback;
m_pAcceptOnIdle = pAcceptOnIdle;

m_nMaximumClientCount = nMaxClients;
m_nTCPPort = nTCPPort;

m_nClientCount = 0;		// No clients
m_nRunFlag = 0;			// Not running

m_nServerSocketTimeout = 500;	// Default timeout is 500 ms (two OnIdle() calls/sec)

m_hListenerThread = NULL;

::InitializeCriticalSection(&m_CritSect);
}


CHost::~CHost()
{

if (m_nRunFlag)
	{
	StopService();
	}



}

void CHost::StopService()
{
if (m_nRunFlag == 0) return;	// Already stopped

m_nRunFlag = 0;					// Ask TCP Listener thread to stop...

//
// If thread is running, stop it.
//

if (m_hListenerThread)
	::WaitForSingleObject(m_hListenerThread, m_nServerSocketTimeout * 2 * m_nMaximumClientCount );

m_hListenerThread = NULL;				// Flag: Thread no longer running

}

///////////////////////////////////////////////////////////////////
//
// Start the TCP Host service.
// 
// Returns 0 on success, or an error code:
//
// 1: BIND failure (the specified port number is already in use)
//
// 2: RUN error (already running the service)
//
// 3: THREAD error (failure to spawn the TCP listener thread)
//
///////////////////////////////////////////////////////////////////

int CHost::StartService()
{

if (m_nRunFlag) return 2;	// RUN error

if (m_hListenerThread) return 2;	// RUN error, already running the thread

//
// Do this before the thread is started -- bind the socket to a port.
//

m_ss.Close();


if ( m_ss.Bind(m_nTCPPort ) == FALSE)
	{
	return 1;
	}

if (m_nMaximumClientCount < 2)		// Single-client (single-threaded user procedure)
	{
	m_hListenerThread = ::CreateThread(NULL,0,CHost::TCP_ListenerThreadProcedure,this,0L,&m_dwThreadID);

	if (m_hListenerThread == NULL)
		return 3;

	return 0;
	}
	else							// Multi-client (multi-threaded user procedure)
	{
	return DoMultithreadHosts();
	}

return 0;		// success (running thread)

}

////////////////////////////////////////////////////////////////////
//
// Multi-threaded TCP Server procedure
//
////////////////////////////////////////////////////////////////////

int CHost::DoMultithreadHosts()
{
m_nClientCount = 0;				// No active connections yet
int i;

m_hClientThreads = new HANDLE[m_nMaximumClientCount + 1];
m_pSockets = new Socket*[m_nMaximumClientCount + 1];

//
// Initialize bank of thread handles
//

for(i=0;i<m_nMaximumClientCount;i++)
	m_hClientThreads[i] = NULL;

//
// Create the Listener thread
//

m_hListenerThread = ::CreateThread(NULL,0,CHost::MT_TCP_ListenerThreadProcedure,this,0L,&m_dwThreadID);

if (m_hListenerThread == NULL)
		return 3;

return 0;	// Running the MT thread
}


DWORD WINAPI CHost::MT_TCP_ClientThreadProcedure(LPVOID lpParm)
{
MT_HostHelper* pmt;
CHost* pHost;
Socket* s;

pmt = (MT_HostHelper*) lpParm;
pHost = pmt->m_pHost;
s = pmt->m_pSocket;

pHost->IncrementClientCount();						// Update client (thread) count in owner

if (pHost->m_pCallback)
		(pHost->m_pCallback)(*s,pHost->m_nRunFlag);	// Call owner's method to deal with the connection
	

s->Close();											// Dump caller

delete s;											// free-up the socket

::CloseHandle(pHost->m_hClientThreads[pmt->m_nIndex] );	// Do this to prevent leaking a handle, since a Wait() method
														// was not used to end this thread...

pHost->m_pSockets[pmt->m_nIndex] = NULL;			// This socket is closed, don't need to close it again.
pHost->m_hClientThreads[ pmt->m_nIndex ] = NULL;	// This thread will be terminated, free up the space
pHost->DecrementClientCount();						// Let owner know we did this

return 0;
}

DWORD WINAPI CHost::MT_TCP_ListenerThreadProcedure(LPVOID lpParam)
{
Socket* s;
CHost* pHost = (CHost*) lpParam;
int i;
DWORD dwThreadID;				// Not really used, just a dummy location for each client's thread ID
pHost->m_nRunFlag = 1;			// Set RUN flag to 1
MT_HostHelper* pmt;



pHost->m_ss.SetSoTimeout(pHost->m_nServerSocketTimeout);	// Set timeout for Accept() method

for(;pHost->m_nRunFlag != 0;)								// Keep running until someone asks us to stop
	{
	if (pHost->GetClientCount() >= pHost->m_nMaximumClientCount)
		{
		::Sleep(pHost->m_nServerSocketTimeout);				// Wait a bit...no available room for new connections
		if (pHost->m_pAcceptOnIdle)
				(pHost->m_pAcceptOnIdle)();					// Call AcceptOnIdle() of owner, no connection
		continue;											// then check again later
		}

	// See if someone is trying to talk with us

	s = pHost->m_ss.Accept();	// Check for incoming TCP connection
	
	if (s != NULL)				// Good connection? See if we can spawn a thread for it
		{
		for(i=0;i<pHost->m_nMaximumClientCount;i++)
			{
			if (pHost->m_hClientThreads[i] == NULL)	// Available slot?
				{
				// Build structure to pass to thread...

				pmt = new MT_HostHelper;
				pmt->m_nIndex = i;
				pmt->m_pHost = pHost;
				pmt->m_pSocket = s;

				pHost->m_pSockets[i] = s;
				pHost->m_hClientThreads[i] = ::CreateThread(NULL,0,CHost::MT_TCP_ClientThreadProcedure,pmt,0L,&dwThreadID);
				break;
				}

			}

		}
		else
		{
		if (pHost->m_pAcceptOnIdle)
				(pHost->m_pAcceptOnIdle)();					// Call AcceptOnIdle() of owner, no connection
		}
	}

//
// Someone asked the service to stop, see if there are any client threads out there
//

if ( pHost->GetClientCount() == 0) return 0;	// No clients, can just return

//
// Wait for any client threads to terminate...force their sockets closed to help the process
// along (will cause socket error on the affected threads)
//

for(i=0;i<pHost->m_nMaximumClientCount;i++)
	{
	if (pHost->m_hClientThreads[i])
		{
		if (pHost->m_pSockets[i] )
				pHost->m_pSockets[i]->Close();
		::WaitForSingleObject(pHost->m_hClientThreads[i],1000);
		}

	}

//
// Free up the data structures in the object (arrays of socket pointers, thread handles)
//

if (pHost->m_pSockets)
	delete[] pHost->m_pSockets;

if (pHost->m_hClientThreads)
	delete[] pHost->m_hClientThreads;

return 0;

}














////////////////////////////////////////////////////////////////////
//
// Single-threaded TCP Server procedure
//
////////////////////////////////////////////////////////////////////

DWORD WINAPI CHost::TCP_ListenerThreadProcedure(LPVOID lpParam)
{
Socket* s;
CHost* pHost = (CHost*) lpParam;

pHost->m_nRunFlag = 1;


pHost->m_ss.SetSoTimeout(pHost->m_nServerSocketTimeout);	// Wait for connection

for(;pHost->m_nRunFlag != 0;)	// Keep running until someone asks us to stop
	{
	s = pHost->m_ss.Accept();	// Check for incoming TCP connection
	
	if (s != NULL)
		{
		if (pHost->m_pCallback)
				(pHost->m_pCallback)(*s,pHost->m_nRunFlag);	// Call owner's method to deal with the connection
		s->Close();
		}
		else
		{
		if (pHost->m_pAcceptOnIdle)
				(pHost->m_pAcceptOnIdle)();					// Call AcceptOnIdle() of owner, no connection
		}
	}

return 0;

}




//////////////////////////////////////////////////////////////////////////////////////
//
// LineInput routine. Returns -1 on timeout, or the number of characters typed. The
// caller must specify the buffer area (buf), and the preferred socket timeout value (nTimeout).
// Timeout defaults to 10,000 ms. Set (nTimeout) to 0 to disable the timeout (infinite timeout).
// 
// nEcho = 0 means no echoplex; nEcho = 1 means normal echo. Echo > 1 means to echo 
// the actual character code in nEcho (may be set to something like '*' for password echo, etc. 
//
// Basic editing (backspace, etc) is supported. The buffer is terminated with '\0'.
//
//////////////////////////////////////////////////////////////////////////////////////

int CHost::LineInput(char* buf, int nLimit, int nEcho, Socket* s, int nTimeout)
{
int nLength;
int nResult;
char c;
int nTotalWaitTime;

if (s == NULL) return -1;	// Error, invalid socket

nLength = 0;

*buf = '\0';

s->SetSoTimeout(nTimeout+100);

for(nTotalWaitTime = 0;;)
	{
	
	nResult = s->Read(&c,1);

	if (nResult == Socket::SOCK_NODATA) 
		{
		Sleep(100);				// kludge, wait 100 ms and try again; ws2 can't really timeout
		nTotalWaitTime += 100;
		if ((nTotalWaitTime > nTimeout) && (nTimeout))
			return -1;		// Error, timeout (no input from user)
		continue;
		}

	if (nResult == Socket::SOCK_TIMEOUT)
		{
		return -1;			// Error, socket timeout; user disconnet?
		}


	if (nResult == Socket::SOCK_ERROR)
		{
		return -1;			// General socket error, return to caller
		}


	nTotalWaitTime = 0;		// Reset wait-timer, user has typed something

	if ((c == 0x08) && (nLength > 0))
		{
		nLength--;
		buf[nLength] = 0;
		if (nEcho)
			s->Write(&c,1);	// write the BS character, if echoplex is on
	
		continue;
		}

	if ((c == 0x08) && (nLength == 0))
		{	
		s->Write("\0x7");		// BEEP! Empty buffer
		continue;
		}


	if (c == 13)			// ENTER?
		{
		buf[nLength] = 0;
		if (nEcho)
			s->Write("\r\n");

		return(nLength);
		}

	if (c < ' ') continue;	// IGNORE other control characters

	if (nLength >= nLimit)
		{
		s->Write("\7");	// BEEP! full buffer
		continue;
		}

	//
	// Put character in buffer

	buf[nLength] = c;
	nLength++;

	if (nEcho==1)		// Echo typed character here
		s->Write(&c,1);

	if (nEcho > 1)
		{
		c = (char) nEcho;
		s->Write(&c,1);	// Echo special character
		}

	}

}



void CHost::IncrementClientCount()
{
::EnterCriticalSection(&m_CritSect);

m_nClientCount++;

::LeaveCriticalSection(&m_CritSect);
}

void CHost::DecrementClientCount()
{
::EnterCriticalSection(&m_CritSect);

m_nClientCount--;

if (m_nClientCount < 0)
	m_nClientCount = 0;

::LeaveCriticalSection(&m_CritSect);
}

int CHost::GetClientCount()
{
int temp;
::EnterCriticalSection(&m_CritSect);

temp = m_nClientCount;

::LeaveCriticalSection(&m_CritSect);

return temp;
}
