// Host.h: interface for the CHost class.
//
// This class implements a TCP host. The host runs in its own
// thread and supports either single or multiple client threads.
//
// The owner supplies the callback for the client thread.
//
// The class also can call a user-defined procedure for timeout
// handling on Accept(), which can be used for a variety of
// functions, such as idle-time broadcasting.
//
// You must link to ws2_32.lib to use this class.
//
// Uses the Socket, InetAddress, and ServerSocket wrapper classes.
//
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_HOST_H__D84D3ED9_533B_4AE8_8362_F97A1788A6F8__INCLUDED_)
#define AFX_HOST_H__D84D3ED9_533B_4AE8_8362_F97A1788A6F8__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include "Socket.h"
#include "DatagramSocket.h"
#include "InetAddress.h"
#include "ServerSocket.h"

#include <windows.h>

class CHost  
{
public:
	int GetClientCount();
	void DecrementClientCount();
	void IncrementClientCount();

	int m_nMaximumClientCount;	// Maximum number of TCP client connections supported. Set to "1" for
								// allowing only one client at a time to connect (single-threaded
								// implementation).

	void StopService();			// Stops the TCP host; dumps any connections, then terminates
								// any outstanding threads. Automatically called by the class
								// destructor.

	int m_nServerSocketTimeout;	// The time period, in ms, that this object waits on the TCP ServerSocket
								// before timing out. What this controls: The time interval between calls
								// to the user's AcceptOnIdle() method.
								//

	int StartService();			// Returns 0 on success, non-zero to indicate some sort of error.
	

	static DWORD WINAPI TCP_ListenerThreadProcedure(LPVOID lpParam);	// ST version (also client handler)

	static DWORD WINAPI MT_TCP_ListenerThreadProcedure(LPVOID lpParam);	// MT version
	static DWORD WINAPI MT_TCP_ClientThreadProcedure(LPVOID lpParam);	// MT client thread
	
	void StripControlChars(char *szBuf);

	CHost(void (*pCallback)(Socket& s, int& nHaltFlag) = NULL, void (*pAcceptOnIdle)() = NULL, int nMaxClients=1, int nTCPPort = 9999);


	int LineInput(char* buf, int nLimit, int nEcho, Socket* s, int nTimeout = 10000 );

	virtual ~CHost();

protected:
	int DoMultithreadHosts();

	HANDLE* m_hClientThreads;	// An array of client thread handles, for multithread mode

	HANDLE m_hListenerThread;	// Handle to TCP listener thread

	DWORD m_dwThreadID;			// ID of TCP listener thread

	int m_nClientCount;			// Current number of TCP clients connected to the virtual host.

	CRITICAL_SECTION m_CritSect;

	Socket** m_pSockets;			// An array of Socket objects, created on each successful connect,
								// and passed one-by-one to the forked threads for each client
								// that gets a connection. Used for: Only to force connections closed
								// if the MT service is discontinued.

	ServerSocket m_ss;			// This server socket is used to listen for TCP connections by
								// the TCP listener thread.

	int m_nRunFlag;				// Set to 1 to allow the service to run, 0 to stop it.

	int m_nTCPPort;				// TCP Port number to listen on; defaults to 9999. Clients will
								// connect to the Host service on this port number.

	void (*m_pCallback)(Socket& s, int& nHaltFlag);

	void (*m_pAcceptOnIdle)();	// Pointer to user's AcceptOnIdle() method, which is called each time
								// the service times out awaiting an incoming connection.

};


//
// This structure is passed to a running client thread in MT mode. It identifies the
// CHost object that owns the thread, the index into the CHost object's internal array of the
// thread, and the socket that the thread will use to communicate.
//

struct MT_HostHelper
{
CHost* m_pHost;
Socket* m_pSocket;
int m_nIndex;
};



#endif // !defined(AFX_HOST_H__D84D3ED9_533B_4AE8_8362_F97A1788A6F8__INCLUDED_)
