// ServerSocket.cpp: implementation of the ServerSocket class.
//
//////////////////////////////////////////////////////////////////////

#include <winsock2.h>	// MUST be first in list
#include "StdAfx.h"
#include "Socket.h"
#include "ServerSocket.h"

/*
#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
#define new DEBUG_NEW
#endif
*/

//
// Global data element to record WSAInit state values
//

WSADATA ServerSocket::m_Data;


//
// Call this method to initialize Winsock 2 BEFORE using
// Sockets calls. It merely wraps around the standard WSAStartup()
// call, storing the data in the class static element m_Data.
//
// Returns: 0 on success, or non-zero on error.
//

int ServerSocket::WSAStartup()
{
return ::WSAStartup(MAKEWORD(1,1), &ServerSocket::m_Data);
}


/////////////////////////////////////////////////////////////////////////////
//
// Constructor: Constructing a ServerSocket object automatically binds the
// ServerSocket to a port. Call the method "GetStatus()" to determine if the
// operation was successful.
//
/////////////////////////////////////////////////////////////////////////////

ServerSocket::ServerSocket(int nPort)
{
int nResult;



m_nAcceptTimeout = 0;			// INFINITE accept() timeout

m_hAcceptEvent = ::WSACreateEvent();


if (nPort == 0)
	{
	m_nStatus = 0;				// Socket status: unbound
	return;						// Can't do anything further
	}


m_PORT = nPort;					// Save the port number we tried to bind to
m_nStatus = StartTCPReceiver(); // Try to listen; returns 0 on failure.

if (m_nStatus == 0) return;		// BIND failure

//
// Set up the listening socket for async operations
//

nResult = ::WSAEventSelect(m_rcvSocket,m_hAcceptEvent,FD_ACCEPT);

//
// Shutdown the socket if unable to hookup event to this socket
//

if (nResult !=0)
	{
	Close();
	}
}

/////////////////////////////////////////////////////////////////////
//
// Manual Bind() procedure:
//
// A ServerSocket can be Closed and "Reopened." When the ServerSocket
// is closed, it is unbound from the port. It can be reopened on
// the same port number or a different port number. This can be
// used, for example, to allow the ServerSocket to refuse connections
// for a specified time period.
//
// Returns: TRUE on success, FALSE on failure
//
/////////////////////////////////////////////////////////////////////

BOOL ServerSocket::Bind(int nPort)
{
int nResult;

if (m_nStatus != 0)
	return FALSE;				// Already open.

if (nPort == 0)
	{
	m_nStatus = 0;				// Socket status: unbound
	return FALSE;				// Can't do anything further
	}


m_PORT = nPort;					// Save the port number we tried to bind to
m_nStatus = StartTCPReceiver(); // Try to listen; returns 0 on failure.

if (m_nStatus == 0) return FALSE;		// BIND failure

//
// Set up the listening socket for async operations
//

nResult = ::WSAEventSelect(m_rcvSocket,m_hAcceptEvent,FD_ACCEPT);

//
// Shutdown the socket if unable to hookup event to this socket
//

if (nResult !=0)
	{
	Close();
	return FALSE;
	}

return TRUE;
}
/////////////////////////////////////////////////////////////////////////////
//
// Destructor: Unbind this ServerSocket from the port to free it up, if it
// is currently occupying the port-number.
//
/////////////////////////////////////////////////////////////////////////////

ServerSocket::~ServerSocket()
{
if (m_nStatus == 0) return;		// *** NOT BOUND, DO NOTHING ***

Close();

m_nStatus = 0;

if (m_hAcceptEvent) ::WSACloseEvent(m_hAcceptEvent);
}

///////////////////////////////////////////////////////////////////////////////////////
//
// StartTCPReceiver()
//
// This method binds the ServerSocket to the specified port number and sets it up
// for listening. It returns 0 on error, 1 on success. After this method has been
// called successfully, the "accept()" method can be used accept incoming connection
// requests from the socket.
//
///////////////////////////////////////////////////////////////////////////////////////

int ServerSocket::StartTCPReceiver()
{
int nStatus;

memset(&m_destSockAddr,0,sizeof(m_destSockAddr));
m_destSockAddr.sin_port = htons(m_PORT);
m_destSockAddr.sin_family = AF_INET;
m_destSockAddr.sin_addr.s_addr = htonl(INADDR_ANY);	// Any IP can talk to this socket

m_rcvSocket = socket(AF_INET, SOCK_STREAM, 0);	// create a TCP socket (SOCK_STREAM)

if (m_rcvSocket == INVALID_SOCKET)
	return 0;									// error creating socket

//
// Bind the TCP socket to the listening port
//

nStatus = bind(m_rcvSocket, (LPSOCKADDR) &m_destSockAddr,sizeof(m_destSockAddr));

if (nStatus == SOCKET_ERROR)
	return 0;									// error binding socket

nStatus = listen(m_rcvSocket, 1);				// LISTEN

if (nStatus == SOCKET_ERROR)
	return 0;


return 1;	// OK, successfully bound this socket to a TCP port
}


////////////////////////////////////////////////////////////////////////////////////////
//
// Method: GetStatus()
//
// Call this method to see if this ServerSocket object is correctly bound to a port.
// Returns 0 if unbound, non-zero if bound.
//
////////////////////////////////////////////////////////////////////////////////////////

int ServerSocket::GetStatus()
{
return m_nStatus;
}

////////////////////////////////////////////////////////////////////////////////////////
//
// Accept(): This method waits for an incoming connection request and returns a new
// Socket object on success, or <NULL> on failure. The method BLOCKS until an incoming
// connection is made.
////////////////////////////////////////////////////////////////////////////////////////

Socket* ServerSocket::Accept()
{
DWORD nResult;

SOCKADDR_IN clientSockAddr;
SOCKET clientSocket;
int addrLen = sizeof(SOCKADDR_IN); // MUST be set to size of structure
int nTimeout;


if (m_nStatus == 0) return NULL;	// Return NULL if this ServerSocket is not yet bound to
								// a port number
nTimeout = m_nAcceptTimeout;

if (m_nAcceptTimeout == 0)
	nTimeout = WSA_INFINITE;

nResult = ::WSAWaitForMultipleEvents(1,	// Number of events
				 &m_hAcceptEvent, // Pointer to array of event handle(s)
				TRUE,				// WaitForAll
				nTimeout,	// Alert Timeout
				FALSE);				// Alertable

if (nResult != WSA_WAIT_EVENT_0) return NULL;		// ** FAILURE **

//
// We received an event of type FD_ACCEPT for the socket; this
// must be reset for the next call to Accept() to function correctly.
// (Otherwise, the event would remain signaled, which would cause
// the waiting function to immediately return each time.)
//

::WSAResetEvent(m_hAcceptEvent);

clientSocket = accept( m_rcvSocket,						// The listening socket
						(LPSOCKADDR) &clientSockAddr,	// Client listening addr
							&addrLen );					// Returns length of client addr

if (clientSocket == INVALID_SOCKET)
			{
			return NULL;	// Error: Failed to accept connection
			}

//
// Successfully received an incoming connection
//

if (clientSocket != INVALID_SOCKET)
			{
			return new Socket( clientSocket, clientSockAddr);
			}


return NULL;	// Failure
}


void ServerSocket::Close()
{
if (m_nStatus == 0)
	return;


shutdown(m_rcvSocket, 2);
closesocket( m_rcvSocket );		// Close the open, listening socket to free it up.

m_nStatus = 0;

}

//
//
// This method sets the accept timeout period. However, its
// action is much more complex than this simple code suggests.
//
// If the m_nAcceptTimeout is 0, then a normal blocking accept
// call is made. If it is non-zero, then an async procedure is
// used in the Accept() method to allow a thread to periodically
// call the method, while efficiently "sleeping" during the
// waiting period.
//
//

void ServerSocket::SetSoTimeout(int nMilli)
{
m_nAcceptTimeout = nMilli;

}


