// Socket.cpp: implementation of the Socket class.
//
//////////////////////////////////////////////////////////////////////

#include <winsock2.h>
#include "stdafx.h"
#include "Socket.h"

#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[]=__FILE__;
//#define new DEBUG_NEW
#endif

//////////////////////////////////////////////////////////////////////
// Symbolic Constants. These are returned by the various I/O
// routines.
//////////////////////////////////////////////////////////////////////

int Socket::SOCK_ERROR = SOCKET_ERROR;
int Socket::SOCK_TIMEOUT = SOCKET_ERROR - 1;
int Socket::SOCK_NODATA = 0;


//
// Global data element to record WSAInit state values
//

WSADATA Socket::m_Data;


//
// Call this method to initialize Winsock 2 BEFORE using
// Sockets calls. It merely wraps around the standard WSAStartup()
// call, storing the data in the class static element m_Data.
//
// Returns: 0 on success, or non-zero on error.
//

int Socket::WSAStartup()
{
return ::WSAStartup(MAKEWORD(1,1), &Socket::m_Data);
}


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Socket::Socket()
{
m_nStatus = 0;	// Not connected to anything
m_nTimeout = 0;
}

Socket::~Socket()
{
Close();
m_nStatus = 0;
}


//////////////////////////////////////////////////////////////////////////////////
//
// Constructor: Socket(InetAddress& dest , int nPort)
//
// This constructor opens the socket and attempts to make an outgoing connection
// to the remote host specified by an InetAddress object. 
// 
// Returns: None, use GetStatus() after constructing to check success, which will return
// zero (0) on failure, or non-zero on success.
//
//////////////////////////////////////////////////////////////////////////////////

Socket::Socket(InetAddress &dest, int nPort)
{
unsigned int addr;		// Used to hold IP address (binary)
int nStatus;

m_nStatus = 0;			// Flag DISCONNECTED initially
m_nTimeout = 0;			// Does nothing except allow value to be returned

m_PORT = nPort;			// Assign the destination port


//
// Get the binary IP address from the InetAddress object
//

addr = dest.GetIP();
::memcpy(&m_destSockAddr.sin_addr , &addr, sizeof(addr));
m_DestIP = addr;


//
// Copy the port number and address family information into the SOCKADDR_IN structure
//


m_destSockAddr.sin_port = htons(m_PORT);			// Destination port
m_destSockAddr.sin_family = AF_INET;		// Family = Internet (IP)

//
// Create the socket
//

m_rcvSocket = ::socket(AF_INET, SOCK_STREAM, 0);

if (m_rcvSocket == INVALID_SOCKET )
	return;

//
// Socket created, try connecting it to the remote system.
//

nStatus = ::connect(m_rcvSocket, (LPSOCKADDR) &m_destSockAddr, sizeof(m_destSockAddr) );

if (nStatus == SOCKET_ERROR)
	{
	closesocket( m_rcvSocket );
	return;
	}


m_nStatus = 1;	// *** Connected ***

return;	
}



//////////////////////////////////////////////////////////////////////////////////
//
// Constructor: Socket(char* szHost , int nPort)
//
// This constructor opens the socket and attempts to make an outgoing connection
// to the remote host specified by "szHost." If the host is specified in dotted-decimal
// notation, the IP address is used directly; if the host is specified using the DNS
// namespace, then DNS resolver services are automatically invoked to resolve the
// hostname into a raw IP address before connecting.
// 
// Returns: None, use GetStatus() after constructing to check success, which will return
// zero (0) on failure, or non-zero on success.
//
//////////////////////////////////////////////////////////////////////////////////
Socket::Socket(char *szHost, int nPort)
{
hostent* remhost;
m_nStatus = 0;			// Flag DISCONNECTED initially
unsigned int addr;		// Used to hold IP address (binary)
int nStatus;

m_nTimeout = 0;			// Does nothing except allow value to be returned

m_PORT = nPort;			// Assign the destination port

//
// If the hostname starts with alpha character, use DNS services to resolve it. The
// first entry in the returned REMHOST structure is used to address the host.
//

if (isalpha(*szHost))
	{

//
// Convert the address data in the structure pointed by *remhost into a 
// SOCKADDR_IN structure for opening the socket.
// The SOCKADDR_IN structure holds the destination address, port, and address
// family identifier.
//
// First, test the data in the *remhost structure.
//

	remhost = ::gethostbyname(szHost);	// Use DNS to lookup the host.

	if (remhost == NULL) return;		// ERROR: unable to do anything further	

	if (remhost->h_addrtype != AF_INET ) return;	// ERROR: Not an IP address, can't process

	if (remhost->h_addr_list[0] == NULL) return; // ERROR: No address data returned

	::memcpy(&m_destSockAddr.sin_addr , remhost->h_addr_list[0], remhost->h_length);
	::memcpy(&m_DestIP, remhost->h_addr_list[0], sizeof(m_DestIP));
	}
	else
//
// Otherwise, just convert the ASCII representation of the host IP into binary.
//
	{
	addr = inet_addr(szHost);
	::memcpy(&m_destSockAddr.sin_addr , &addr, sizeof(addr));
	m_DestIP = addr;
	}



//
// Copy the port number and address family information into the SOCKADDR_IN structure
//


m_destSockAddr.sin_port = htons(m_PORT);			// Destination port
m_destSockAddr.sin_family = AF_INET;		// Family = Internet (IP)

//
// Create the socket
//

m_rcvSocket = ::socket(AF_INET, SOCK_STREAM, 0);

if (m_rcvSocket == INVALID_SOCKET )
	return;

//
// Socket created, try connecting it to the remote system.
//

nStatus = ::connect(m_rcvSocket, (LPSOCKADDR) &m_destSockAddr, sizeof(m_destSockAddr) );

if (nStatus == SOCKET_ERROR)
	{
	closesocket( m_rcvSocket );
	return;
	}


m_nStatus = 1;	// *** Connected ***

return;	
}


///////////////////////////////////////////////////////////////////////////////////
//
// Constructor version 2
//
// Function: This constructor can be called with a valid SOCKET handle and SOCKADDR_IN 
// structure to establish an open connection. It assumes that the passed socket is
// actually open and usable. This constructor is used by the Accept() method of the
// ServerSocket class. Normally it should not be used otherwise.
//
///////////////////////////////////////////////////////////////////////////////////

Socket::Socket(SOCKET &sock, SOCKADDR_IN &addr)
{


m_rcvSocket = sock;		// Copy the HANDLE to the caller's socket
m_destSockAddr = addr;	// Copy the ADDRESS data of the caller's socket
::memcpy(&m_DestIP,&m_destSockAddr.sin_addr,sizeof(m_DestIP));
m_nStatus = 1;			// ** Connected (assumed, this could be wrong.) **
}

//////////////////////////////////////////////////////////////////////////////////
//
// Close(): Disconnects this Socket, no linger. The connection is closed and
// the socket is no longer useable. To reopen a connection, create a new Socket
// object.
//////////////////////////////////////////////////////////////////////////////////

void Socket::Close()
{
if (m_nStatus == 0) return; // Already closed, or was never open

shutdown(m_rcvSocket, 2);
closesocket( m_rcvSocket );

m_nStatus = 0;				// Flag: Now closed

}

//////////////////////////////////////////////////////////////////////////////////////
//
// Writing functions. These write various forms of data to the socket. They all return
// ZERO (or Socket::SOCK_NODATA ) on error, or the number of bytes written.
//
//////////////////////////////////////////////////////////////////////////////////////

int Socket::Write(char *szData)
{
int nNumSent;

if (m_nStatus == 0) return 0;


nNumSent = ::send(m_rcvSocket,szData,strlen(szData),0);

return nNumSent;
}

int Socket::Write(void *buf, int nBytes)
{
int nNumSent;

if (m_nStatus == 0) return 0;

nNumSent = ::send(m_rcvSocket,(char*) buf,nBytes, 0);

return nNumSent;
}

////////////////////////////////////////////////////////////
//
// SetSoTimeout()
//
// Set the timeout, for transmit and receive operations, on this
// Socket object. The default mode of
// the socket is the same as in Winsock2 -- the socket runs
// synchronously and blocks when no data is available.
// Setting this function limits the blocking behavior for ALL
// receive methods (below).
//
// The timeout value is in milliseconds.
//
////////////////////////////////////////////////////////////

void Socket::SetSoTimeout(int nMilli)
{
if (m_nStatus == 0) return; // Not open

m_nTimeout = nMilli;

int nResult;

nResult = ::setsockopt(m_rcvSocket,SOL_SOCKET,SO_RCVTIMEO,
		(char*) &nMilli, sizeof(int));

nResult = ::setsockopt(m_rcvSocket,SOL_SOCKET,SO_SNDTIMEO,
		(char*) &nMilli, sizeof(int));



}

/////////////////////////////////////////////////////////////
//
// Receive functions
//
// These functions receive data from an open Socket. If no
// data is available, zero is returned; SOCK_ERROR is returned
// if a problem is detected. These functions are blocking, meaning
// that they will wait indefinitely if no data arrives. To
// alter this behavior, use SetSoTimeout() to establish a
// timeout value for the receiver.
//
// If no data is available, SOCK_NODATA is returned (if the timeout
// option is set). Due to the WS2 implemention, normally SOCK_TIMEOUT
// won't be returned, so to check for a real timeout, wait a short
// time, then call the receive method again.
//
///////////////////////////////////////////////////////////////


//
// Generic read. Attempt to read the number of bytes asked
// for by the owner of the object, returning the actual number
// of bytes read. Will NOT block if any data is received -- will
// return the actual number of bytes read.
//

int Socket::Read(void *lpBuf, int nBytesToRead)
{
int nStatus;

if (m_nStatus == 0) return SOCK_ERROR;

nStatus = ::recv(m_rcvSocket,(char*) lpBuf,nBytesToRead,0);

//
// If the returned status is not an error, return it as the
// number of bytes read. This path can also return zero, as
// SOCK_NODATA (0).
//

if (nStatus != SOCKET_ERROR)
	return nStatus;

//
// An error occurred; use WSAGetLastError() and convert to an
// error result code.
//

nStatus = WSAGetLastError();

if (nStatus == WSAEWOULDBLOCK) return SOCK_NODATA;	// No data, socket not ready

if (nStatus == WSAETIMEDOUT ) return SOCK_TIMEOUT;

return -1; //SOCK_ERROR;
}

//
// Buffered read. Attempt to read bytes until the caller's
// buffer is full, OR a <cTerminator> character is detected.
// In either case, the buffer is terminated with a <NUL> (0x00).
//
// The <cTerminator> value defaults to 0x0d, <CR>.
//
// Returns: The number of characters read into the buffer
// (including the terminator, which is written into the
// buffer), OR an error codes as above.
//
//

int Socket::ReadLine(char *szBuf, int nMaxChars, char cTerminator)
{
int nRead = 0;	// Number of characters read
int nResult;

	*szBuf = 0;					// Terminate buffer (buffer will be
								// empty if no data received).

	for(;nRead < (nMaxChars-1);)
		{
		nResult = Read( szBuf, 1 ); // Read one character from socket
	
		if ((nResult == SOCK_ERROR) || (nResult == SOCK_TIMEOUT))
			{
			return nResult;
			}

		// Data was read OK. Terminate with \0 and search for <cTerminator>



		szBuf[1] = 0;	// Terminate buffer
		nRead++;		// Increment read-character counter

		if (*szBuf == cTerminator)
			return nRead;

		szBuf++;		// Increment buffer-pointer
		}
//
// Fall thru to here when the buffer is full
//

return nRead;
}


int Socket::GetStatus()
{
return m_nStatus;
}

int Socket::GetLocalPort()
{
SOCKADDR_IN sa;
char buf[64];
int nResult;
int nNamLen = sizeof(buf)-1;
buf[0]=0;

if (m_nStatus == 0) return 0;

nResult = ::getsockname(m_rcvSocket,(SOCKADDR*) &sa,&nNamLen );

return sa.sin_port;

}

int Socket::GetSoTimeout()
{
return m_nTimeout;
}

InetAddress* Socket::GetLocalAddress()
{
SOCKADDR_IN sa;
char buf[64];
int nResult;
int nNamLen = sizeof(buf)-1;
buf[0]=0;
unsigned int nAddr;

if (m_nStatus == 0) return 0;

nResult = ::getsockname(m_rcvSocket,(SOCKADDR*) &sa,&nNamLen );

::memcpy(&nAddr,&sa.sin_addr,sizeof(nAddr));

return new InetAddress(nAddr);
}

InetAddress* Socket::GetRemoteAddress()
{

if (m_nStatus == 0) return NULL;


return new InetAddress(m_DestIP);
}

///////////////////////////////////////////////////////////////
//
// GetSO_SNDBUF()
//
// This method returns the buffer size of the TCP socket. Transmitting
// more than this amount of data without checking that the data is
// actually sent will cause data to be lost. Use SetSO_SNDBUF() to
// change the TCP buffer size.
//
// Returns: The socket's buffer size, in bytes.
//
//////////////////////////////////////////////////////////////

int Socket::GetSO_SNDBUF()
{
int value = 0;
int nsize = sizeof(int);

::getsockopt(m_rcvSocket,SOL_SOCKET,SO_SNDBUF,
		(char*) &value, &nsize );

return value;
}

void Socket::SetSO_SNDBUF(int nBufSize)
{
if(m_nStatus)
	::setsockopt(m_rcvSocket,SOL_SOCKET,SO_SNDBUF,
			(char*) &nBufSize,sizeof(int));


}
